"""
=============================================================================
SCRIPT: AnthrOS_Infrastructure_Compiler_Perplexity.py
CODENAME: "The Tank" (Enhanced Logging Edition)
PROJECT: Project AnthroHeart
AUTHOR: Thomas B. Sweet aka Anthro Teacher/Cio
UPDATES:
- Converted to Perplexity API with Claude 3.5 Sonnet
- Added Windows Long Path support (\\?\\ prefix)
- Added Error Skipping (Script won't crash on one bad file)
- Forces Absolute Paths to prevent "Path not found" errors
- Enhanced structured logging to console and file
- Per-file success/error tracking
- Machine-readable JSON run report for debugging
=============================================================================
"""

import os
import shutil
import json
import random
import time
import textwrap
import logging
from pathlib import Path
from datetime import datetime

print("\n[ SYSTEM ] Booting AnthrOS Infrastructure (Robust Mode, Logged)...")

try:
    from openai import OpenAI
    from PIL import Image, ImageDraw, ImageFont
    from fpdf import FPDF
    import requests
    import mutagen
    from mutagen.id3 import ID3, TIT2, TPE1, TALB, TCON, COMM, APIC, TXXX
    from mutagen.wave import WAVE
    from pypdf import PdfReader, PdfWriter
except ImportError as e:
    print(f"\n[ CRITICAL ] Missing Library: {e}")
    print("Run: pip install openai pillow mutagen pypdf fpdf requests")
    exit()

# ==========================================
# LOGGING SETUP
# ==========================================

LOG_DIR = Path("anthros_logs")
LOG_DIR.mkdir(exist_ok=True)

log_file = LOG_DIR / "anthros_compiler.log"
logging.basicConfig(
    level=logging.INFO,
    format="%(asctime)s [%(levelname)s] %(message)s",
    handlers=[
        logging.FileHandler(log_file, encoding="utf-8"),
        logging.StreamHandler()
    ],
)

logger = logging.getLogger("AnthrOS")

RUN_REPORT_PATH = LOG_DIR / "anthros_run_report.json"

# for run statistics
RUN_STATS = {
    "start_time": datetime.utcnow().isoformat() + "Z",
    "end_time": None,
    "total_files_scanned": 0,
    "total_files_copied": 0,
    "total_files_patched": 0,
    "errors": [],
    "ai_calls": 0,
    "hollywood": {
        "bible_generated": False,
        "pitch_generated": False
    }
}

def strip_markdown_json(content: str) -> str:
    if "```json" in content:
        return content.split("```json", 1)[1].split("```", 1)[0].strip()
    if "```" in content:
        return content.split("```", 1)[1].split("```", 1)[0].strip()
    return content

def add_error(context, fpath, exc):
    RUN_STATS["errors"].append({
        "context": context,
        "file": str(fpath) if fpath else None,
        "error": repr(exc)
    })
    logger.error(f"[{context}] {fpath}: {exc}")

def write_run_report():
    RUN_STATS["end_time"] = datetime.utcnow().isoformat() + "Z"
    try:
        with open(RUN_REPORT_PATH, "w", encoding="utf-8") as f:
            json.dump(RUN_STATS, f, indent=2)
        logger.info(f"[REPORT] Run report written to {RUN_REPORT_PATH}")
    except Exception as e:
        logger.error(f"[REPORT] Failed to write run report: {e}")

# ==========================================
# 2. CONFIGURATION MATRIX
# ==========================================

NEXUS_URL = "https://nexus.anthroentertainment.com"

BRANDING = {
    "project_name": "Project AnthroHeart",
    "version": "AnthrOS 1.0",
    "author": "Thomas B. Sweet",
    "website": "anthroentertainment.com",
    "warlock_artist_1": "certaintlyart.anthroentertainment.com",
    "warlock_artist_2": "orbnorsolyaart.anthroentertainment.com",
}

ECOSYSTEM = {
    "WordFinder": "wordfinder.anthroentertainment.com",
    "Tarot": "tarot.anthroentertainment.com",
    "ButCanHeDraw": "butcanhedraw.anthroentertainment.com",
    "Servitor": "servitorconnect.anthroentertainment.com",
    "History": "myfirstcomputer.anthroentertainment.com",
    "MagnumOpus": "magnumopus.anthroentertainment.com",
}

ASSETS = {
    "blueheartfoxicon.png": "http://foxicon.anthroentertainment.com",
    "dogicon.png": "http://dogicon.anthroentertainment.com",
    "anthronessgemicon.png": "http://gemicon.anthroentertainment.com",
    "trinityicon.png": "http://trinityicon.anthroentertainment.com",
    "anthroheartreceipt.png": "http://receipt.anthroentertainment.com",
}

OUTPUT_DIR = "Updated_AnthroHeart_Archive"
HOLLYWOOD_DIR = "Hollywood_Pitch_Package"
CACHE_DIR = "anthros_assets_cache"

ZINGERS = [
    "Fur is temporary, style is forever.",
    "Paws off the merchandise.",
    "Barking up the right tree.",
    "404: Human not found.",
    "Digital hearts, analog souls.",
    "Welcome to the pack.",
    "The Warlock sees all.",
]

# ==========================================
# 3. AUTHENTICATION (PERPLEXITY)
# ==========================================

def get_api_key():
    try:
        key_path = Path("..") / "Perplexity_Key.txt"
        with open(key_path, "r") as f:
            return f.read().strip()
    except Exception as e:
        logger.critical(f"[AUTH] API Key missing or unreadable: {e}")
        print("[ ERROR ] API Key missing. Check '../Perplexity_Key.txt'")
        exit()

client = OpenAI(
    api_key=get_api_key(),
    base_url="https://api.perplexity.ai"
)

# ==========================================
# 4. ASSET ACQUISITION
# ==========================================

def acquire_assets():
    logger.info("[NET] Connecting to AnthrOS Subdomains...")
    if not os.path.exists(CACHE_DIR):
        os.mkdir(CACHE_DIR)

    downloaded = {}
    for filename, url in ASSETS.items():
        local = Path(CACHE_DIR) / filename
        try:
            headers = {'User-Agent': 'AnthrOS-Compiler/1.0'}
            r = requests.get(url, headers=headers, timeout=15, allow_redirects=True)
            if r.status_code == 200:
                with open(local, 'wb') as f:
                    f.write(r.content)
                downloaded[filename] = local
                logger.info(f"[NET] Downloaded asset: {filename} from {url}")
            else:
                if local.exists():
                    downloaded[filename] = local
                    logger.warning(f"[NET] HTTP {r.status_code} for {url}, using cached {filename}")
                else:
                    logger.warning(f"[NET] Failed {url} (status {r.status_code}), no local cache.")
        except Exception as e:
            if local.exists():
                downloaded[filename] = local
                logger.warning(f"[NET] Exception for {url}, using cached {filename}: {e}")
            else:
                add_error("ASSET_DOWNLOAD", filename, e)
    logger.info(f"[NET] Total assets resolved: {len(downloaded)}")
    return downloaded

# ==========================================
# 5. AI INTELLIGENCE (CLAUDE 3.5 SONNET)
# ==========================================

def analyze_file(filename):
    prompt = f"""
Analyze file: "{filename}".
Context: "The Warlock Name" (Art Copyright, Story CC0). "Project AnthroHeart" (CC0).
Return ONLY a valid JSON object with these keys:
{{"is_warlock": bool, "type": "art/audio/text/pdf/other", "artist": "generic/certaintly/orbnorsolya"}}
"""

    for attempt in range(1, MAX_RETRIES + 1):
        try:
            resp = client.chat.completions.create(
                model="sonar-pro",
                messages=[{"role": "user", "content": prompt}]
            )

            RUN_STATS["ai_calls"] += 1
            content = resp.choices[0].message.content
            content = strip_markdown_json(content)
            return json.loads(content)

        except Exception as e:
            msg = str(e)

            if "429" in msg or "Too Many Requests" in msg:
                delay = BASE_DELAY * (2 ** (attempt - 1))
                jitter = random.uniform(0.2, 0.8)
                wait = delay + jitter

                logger.warning(
                    f"[AI_RATE_LIMIT] {filename} – retry {attempt}/{MAX_RETRIES} in {wait:.1f}s"
                )
                time.sleep(wait)
                continue

            # Non-rate-limit error
            add_error("AI_ANALYZE", filename, e)
            break

    # Fallback: no AI result
    logger.warning(f"[AI_FALLBACK] Using defaults for {filename}")
    return {"is_warlock": False, "type": "other", "artist": "generic"}

# ==========================================
# 6. FILE PATCHERS
# ==========================================

def patch_image(fpath, ai_data):
    try:
        img = Image.open(fpath)
        if ai_data.get("is_warlock") and ai_data.get("type") == "art":
            color = (100, 0, 0, 220)
            artist = BRANDING['warlock_artist_1']
            if ai_data.get("artist") == "orbnorsolya":
                artist = BRANDING['warlock_artist_2']
            txt = f"© COPYRIGHTED ART | Artist: {artist}"
        else:
            color = (0, 0, 0, 180)
            txt = f"PUBLIC DOMAIN [CC0] | Nexus: {NEXUS_URL}"

        img = img.convert("RGBA")
        w, h = img.size
        draw = ImageDraw.Draw(img)
        draw.rectangle([(0, h-40), (w, h)], fill=color)
        try:
            font = ImageFont.truetype("arial.ttf", 16)
        except Exception:
            font = ImageFont.load_default()
        draw.text((20, h-30), f"{txt}", fill=(255, 255, 255), font=font)

        if fpath.suffix.lower() in ['.jpg', '.jpeg']:
            img = img.convert("RGB")
        img.save(fpath, quality=95)
        logger.info(f"[PATCH_IMG] Patched image metadata: {fpath}")
        RUN_STATS["total_files_patched"] += 1
    except Exception as e:
        add_error("PATCH_IMAGE", fpath, e)

def patch_audio(fpath, ai_data, icons):
    try:
        zinger = random.choice(ZINGERS)
        receipt = ASSETS['anthroheartreceipt.png']
        valid = {k: v for k, v in icons.items() if 'receipt' not in k}
        icon_path = random.choice(list(valid.values())) if valid else None
        ext = fpath.suffix.lower()

        if ext == '.mp3':
            try:
                audio = ID3(fpath)
            except Exception:
                audio = ID3()
            audio.add(TIT2(encoding=3, text=fpath.stem))
            audio.add(TPE1(encoding=3, text=BRANDING['author']))
            audio.add(TXXX(encoding=3, desc='AnthrOS_Nexus', text=NEXUS_URL))
            audio.add(TXXX(encoding=3, desc='Receipt', text=f"$21k Verified | {receipt}"))
            audio.add(TXXX(encoding=3, desc='Zinger', text=zinger))
            audio.add(COMM(encoding=3, desc='License', text="Public Domain [CC0]"))

            if icon_path:
                with open(icon_path, 'rb') as art:
                    audio.add(APIC(3, 'image/png', 3, 'Cover', art.read()))
            audio.save(fpath)
            logger.info(f"[PATCH_AUDIO] MP3 tagged: {fpath}")
            RUN_STATS["total_files_patched"] += 1

        elif ext == '.wav':
            audio = WAVE(fpath)
            audio.add_tags()
            audio.tags["ICMT"] = f"AnthrOS Nexus: {NEXUS_URL} | Cost: $21k | License: CC0"
            audio.save(fpath)
            logger.info(f"[PATCH_AUDIO] WAV tagged: {fpath}")
            RUN_STATS["total_files_patched"] += 1
    except Exception as e:
        add_error("PATCH_AUDIO", fpath, e)

def patch_docs(fpath, ai_data):
    try:
        ext = fpath.suffix.lower()
        receipt = ASSETS['anthroheartreceipt.png']

        if ext in ['.txt', '.md']:
            footer = f"\n\n---\nAnthrOS Nexus: {NEXUS_URL}\nVerified Cost: $21k | {receipt}\n"
            with open(fpath, "a", encoding="utf-8") as f:
                f.write(footer)
            logger.info(f"[PATCH_DOC] Text/MD footer added: {fpath}")
            RUN_STATS["total_files_patched"] += 1

        elif ext == '.pdf':
            reader = PdfReader(fpath)
            writer = PdfWriter()
            for p in reader.pages:
                writer.add_page(p)
            writer.add_metadata({
                "/Author": BRANDING['author'],
                "/AnthrOS_Nexus": NEXUS_URL,
                "/Receipt": f"$21k Verified | {receipt}"
            })
            with open(fpath, "wb") as f:
                writer.write(f)
            logger.info(f"[PATCH_DOC] PDF metadata added: {fpath}")
            RUN_STATS["total_files_patched"] += 1
    except Exception as e:
        add_error("PATCH_DOC", fpath, e)

# ==========================================
# 7. HOLLYWOOD GENERATOR
# ==========================================

class HollywoodDoc(FPDF):
    def __init__(self, title, icon_path=None):
        super().__init__()
        self.doc_title = title
        self.icon_path = icon_path
        self.set_auto_page_break(auto=True, margin=20)

    def header(self):
        if self.icon_path:
            try:
                self.image(self.icon_path, 10, 8, 15)
            except Exception:
                pass
        self.set_font('Arial', 'B', 10)
        self.cell(0, 10, f"AnthrOS | {self.doc_title}", 0, 0, 'R')
        self.ln(20)

    def cover(self, title, sub):
        self.add_page()
        self.set_fill_color(10, 10, 15)
        self.rect(0, 0, 210, 297, 'F')
        if self.icon_path:
            try:
                self.image(self.icon_path, x=80, y=60, w=50)
            except Exception:
                pass
        self.set_y(140)
        self.set_text_color(255, 255, 255)
        self.set_font('Arial', 'B', 24)
        self.multi_cell(0, 10, title.upper(), 0, 'C')
        self.ln(10)
        self.set_font('Arial', '', 12)
        self.set_text_color(0, 240, 255)
        self.multi_cell(0, 10, sub, 0, 'C')
        self.add_page()
        self.set_text_color(0, 0, 0)

    def section(self, title, body):
        self.set_font('Arial', 'B', 14)
        self.set_text_color(0, 0, 0)
        self.cell(0, 10, title, 0, 1)
        self.set_font('Times', '', 12)
        self.set_text_color(50, 50, 50)
        clean = body.encode('latin-1', 'replace').decode('latin-1')
        self.multi_cell(0, 6, clean)
        self.ln(5)

def generate_hollywood(files, icons):
    logger.info("[HOLLYWOOD] Compiling Assets...")
    dest = Path(OUTPUT_DIR) / HOLLYWOOD_DIR
    if not dest.exists():
        dest.mkdir(parents=True)

    defaults = {
        "logline": "25 Years of Anthro Legacy.",
        "hook": "The first public domain Anthro universe.",
        "lore": "",
        "characters": "",
        "viral_loop": ""
    }

    # 1. BIBLE
    try:
        r = client.chat.completions.create(
            model="sonar-pro",
            messages=[{"role": "user", "content": "Write TV Bible for Project AnthroHeart. Return JSON with keys: logline, lore, characters."}]
        )
        RUN_STATS["ai_calls"] += 1
        content = r.choices[0].message.content
        # Safely strip markdown code fences if present
        content = strip_markdown_json(content)

        data = json.loads(content)
    except Exception as e:
        add_error("HOLLYWOOD_BIBLE_AI", None, e)
        data = defaults

    try:
        pdf = HollywoodDoc("Series Bible", icons.get("trinityicon.png"))
        pdf.cover("AnthroHeart Bible", f"Nexus: {NEXUS_URL}")
        pdf.section("Logline", data.get("logline", ""))
        pdf.section("Lore", data.get("lore", ""))
        pdf.section("Characters", data.get("characters", ""))
        out_path = dest / "AnthroHeart_Series_Bible.pdf"
        pdf.output(out_path)
        RUN_STATS["hollywood"]["bible_generated"] = True
        logger.info(f"[HOLLYWOOD] Bible generated at {out_path}")
    except Exception as e:
        add_error("HOLLYWOOD_BIBLE_PDF", None, e)

    # 2. PITCH DECK
    try:
        r = client.chat.completions.create(
            model="sonar-pro",
            messages=[{"role": "user", "content": "Write Pitch Deck for Project AnthroHeart. Return JSON with keys: hook, demographics, viral_loop."}]
        )
        RUN_STATS["ai_calls"] += 1
        content = r.choices[0].message.content
        # Safely strip markdown code fences if present
        content = strip_markdown_json(content)

        data = json.loads(content)
    except Exception as e:
        add_error("HOLLYWOOD_PITCH_AI", None, e)
        data = defaults

    try:
        pdf = HollywoodDoc("Pitch Deck", icons.get("blueheartfoxicon.png"))
        pdf.cover("Investment Deck", "The $21k Verified Legacy")
        pdf.section("The Hook", data.get("hook", ""))
        pdf.section("Viral Loop", data.get("viral_loop", ""))
        out_path = dest / "Visual_Pitch_Deck.pdf"
        pdf.output(out_path)
        RUN_STATS["hollywood"]["pitch_generated"] = True
        logger.info(f"[HOLLYWOOD] Pitch Deck generated at {out_path}")
    except Exception as e:
        add_error("HOLLYWOOD_PITCH_PDF", None, e)

# ==========================================
# 8. MAIN EXECUTION (ROBUST MODE)
# ==========================================

def main():
    logger.info("===================================================")
    logger.info(" ANTHROS INFRASTRUCTURE COMPILER (ROBUST, LOGGED)")
    logger.info("===================================================")

    try:
        icons = acquire_assets()
        src = Path(".").resolve()
        dest = (src / OUTPUT_DIR).resolve()
        if not dest.exists():
            dest.mkdir()

        files = []
        logger.info("[SCANNING] Indexing Files...")
        for root, dirs, f_list in os.walk(src):
            if OUTPUT_DIR in root or CACHE_DIR in root or str(LOG_DIR) in root:
                continue
            for f in f_list:
                if f.endswith(".py") or f.endswith(".txt") or f.startswith("."):
                    continue
                files.append(Path(root) / f)

        RUN_STATS["total_files_scanned"] = len(files)
        logger.info(f"[SCANNING] Found {len(files)} files to process.")

        logger.info("[COMPILING] Copying + stamping Nexus link...")
        for i, fpath in enumerate(files):
            if RUN_STATS["total_files_scanned"] > 0 and i % max(1, RUN_STATS["total_files_scanned"] // 10) == 0:
                percent = int(i / RUN_STATS["total_files_scanned"] * 100)
                logger.info(f"[PROGRESS] {percent}% ({i}/{RUN_STATS['total_files_scanned']})")

            try:
                rel_path = fpath.relative_to(src)
                target = dest / rel_path

                target_str = str(target.resolve())
                if os.name == 'nt' and len(target_str) > 240:
                    target_str = "\\\\?\\" + target_str

                os.makedirs(os.path.dirname(target_str), exist_ok=True)
                shutil.copy2(fpath, target_str)
                RUN_STATS["total_files_copied"] += 1

                ai_data = analyze_file(fpath.name)
                ext = fpath.suffix.lower()
                target_path_obj = Path(target_str)

                if ext in ['.jpg', '.png']:
                    patch_image(target_path_obj, ai_data)
                elif ext in ['.mp3', '.wav']:
                    patch_audio(target_path_obj, ai_data, icons)
                elif ext in ['.txt', '.pdf', '.md']:
                    patch_docs(target_path_obj, ai_data)

            except Exception as e:
                add_error("MAIN_FILE_PROCESS", fpath, e)
                continue

        logger.info("[DONE] Metadata injection complete.")
        generate_hollywood(files, icons)

        logger.info("===================================================")
        logger.info(" INFRASTRUCTURE COMPILED.")
        logger.info("===================================================")

    finally:
        write_run_report()

if __name__ == "__main__":
    main()